﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Windows.Forms;

using MicroRWD.MFIC;

namespace MicroRWD.UI.MFIC
{
    public partial class WriteToMemoryPanel : UserControl
    {
        #region Public Properties

        public byte Address
        {
            get
            {
                if (ListView != null)
                {
                    if (ListView.SelectedIndices.Count > 0)
                    {
                        return (byte)ListView.SelectedIndices[0];
                    }
                    else
                    {
                        return 0;
                    }
                }
                else
                {
                    return 0;
                }
            }
            set
            {
                if (ListView != null)
                {
                    ListView.Items[value].Selected = true;
                }
            }
        }

        private event EventHandler addressChanged;
        public event EventHandler Address_Changed { add { addressChanged += value; } remove { addressChanged -= value; } }

        public event EventHandler Click_CloseButton { add { closeButton.Click += value; } remove { closeButton.Click -= value; } }

        public event EventHandler Click_WriteButton { add { writeButton.Click += value; } remove { writeButton.Click -= value; } }

        private int count;
        public int Count
        {
            get
            {
                return count;
            }
            set
            {
                count = value;
            }            
        }
        
        private byte[] data;
        public byte[] Data
        {
            get
            {
                return data;
            }
            set
            {
                data = value;
            }
        }

        public string Description
        {
            get
            {
                if (ListView != null)
                {
                    return ListView.Items[Address].SubItems[DescriptionSubItem].Text;
                }
                else
                {
                    return "";
                }
            }
        }

        public int DescriptionSubItem { get; set; }

        public ListView ListView { get; set; }

        public bool Modified { get; set; }

        #endregion


        #region Constructor

        public WriteToMemoryPanel()
        {
            InitializeComponent();

            // Initialise state
            Count = 1;
            DescriptionSubItem = 1;
            Modified = false;
        }

        #endregion


        #region Public Methods

        public void UpdateView()
        {
            this.UIThread(() => RefreshView());
        }

        #endregion


        #region Private Methods

        private string FormatAscii()
        {
            string result = "";

            for (int i = 0; i < Count; ++i)
            {
                if (i > 0)
                {
                    result += ' ';
                }

                if ((Data != null) && (i < Data.Length) && (Data[i] >= 32) && (Data[i] <= 126))
                {
                    result += (char)Data[i];
                }
                else
                {
                    result += '.';
                }
            }

            return result;
        }

        private string FormatHex()
        {
            string result = "";

            for (int i = 0; i < Count; ++i)
            {
                if (i > 0)
                {
                    result += ' ';
                }

                if ((Data != null) && (i < Data.Length))
                {
                    result += ((byte)Data[i]).ToString("X2");
                }
                else
                {
                    result += "XX";
                }
            }

            return result;
        }

        private byte HexToByte(char _c)
        {
            byte result = 0;

            if ((_c >= '0') && (_c <= '9'))
            {
                result = (byte)(_c - '0');
            }
            else if ((_c >= 'A') && (_c <= 'F'))
            {
                result = (byte)(10 + (_c - 'A'));
            }
            else if ((_c >= 'a') && (_c <= 'f'))
            {
                result = (byte)(10 + (_c - 'a'));
            }

            return result;
        }

        private bool IsHexDigit(char _c)
        {
            if (((_c >= '0') && (_c <= '9')) || ((_c >= 'A') && (_c <= 'F')) || ((_c >= 'a') && (_c <= 'f')))
            {
                return true;
            }

            return false;
        }

        private void RefreshView()
        {
            addressTextBox.Text = Address.ToString("X2");
            descriptionTextBox.Text = Description;
            int cur = hexTextBox.SelectionStart;
            hexTextBox.MaxLength = Count * 3 - 1;
            hexTextBox.Text = FormatHex();
            hexTextBox.SelectionStart = cur;
            hexTextBox.SelectionLength = 1;
            cur = asciiTextBox.SelectionStart;
            asciiTextBox.MaxLength = Count * 2 - 1;
            asciiTextBox.Text = FormatAscii();
            asciiTextBox.SelectionStart = cur;
            asciiTextBox.SelectionLength = 1;
            if (!Modified)
            {
                hexTextBox.ForeColor = Color.Black;
                asciiTextBox.ForeColor = Color.Black;
            }
            else
            {
                hexTextBox.ForeColor = Color.Red;
                asciiTextBox.ForeColor = Color.Red;
            }
        }

        private void FireAddressChangedEvent()
        {
            if (addressChanged != null)
            {
                addressChanged(this, new EventArgs());
            }
        }

        #region Event Handlers

        private void WriteToMemoryPanel_Load(object sender, EventArgs e)
        {
            if (!DesignMode && (Program.Controller != null))
            {
                // Refresh view
                RefreshView();
            }
        }

        private void nextButton_Click(object sender, EventArgs e)
        {
            // Increment address if not already at last
            if (Address < ListView.Items.Count - 1)
            {
                ++Address;
                Data = null;
                Modified = false;

                // Refresh view
                RefreshView();

                // Notify listeners
                FireAddressChangedEvent();
            }
        }

        private void prevButton_Click(object sender, EventArgs e)
        {
            // Decrement address if not already at first
            if (Address > 0)
            {
                --Address;
                Data = null;
                Modified = false;

                // Refresh view
                RefreshView();

                // Notify listeners
                FireAddressChangedEvent();        
            }
        }

        private void hexTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (Count > 0)
            {
                // Process backspace and non hex digit key presses
                if (e.KeyChar == (char)8)
                {
                    // XX_XX_XX_XX
                    hexTextBox.SelectionStart = (hexTextBox.SelectionStart > 0) ? hexTextBox.SelectionStart - 1 : 0;

                    // Refresh view
                    RefreshView();
                }
                else if (IsHexDigit(e.KeyChar))
                {
                    // XX_XX_XX_XX
                    int cur = hexTextBox.SelectionStart;
                    if (((cur + 1) % 3) == 0)
                    {
                        ++cur;
                    }
                    int pos = Math.Min(cur / 3, Count - 1);

                    if (Data == null)
                    {
                        Data = new byte[Count];
                        Modified = true;
                    }

                    // Establish edited value
                    byte v = Data[pos];
                    if ((cur % 3) == 0)
                    {
                        // hi byte
                        v = (byte)(((HexToByte(e.KeyChar) << 4) & 0xf0) | (Data[pos] & 0x0f));
                    }
                    else
                    {
                        // lo byte
                        v = (byte)((Data[pos] & 0xf0) | (HexToByte(e.KeyChar) & 0x0f));
                    }

                    // Adjust cursor XX_XX_XX_XX
                    ++cur;
                    if (((cur + 1) % 3) == 0)
                    {
                        ++cur;
                    }

                    hexTextBox.SelectionStart = Math.Min(cur, hexTextBox.MaxLength - 1);

                    // Update data bytes (if changed)
                    if (Data[pos] != v)
                    {
                        Data[pos] = v;
                        Modified = true;
                    }

                    // Refresh view
                    RefreshView();
                }
            }

            e.Handled = true;
        }

        private void asciiTextBox_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (Count > 0)
            {
                // Process backspace and ascii key presses
                if (e.KeyChar == (char)8)
                {
                    // ._._._.
                    asciiTextBox.SelectionStart = (asciiTextBox.SelectionStart > 0) ? asciiTextBox.SelectionStart - 1 : 0;

                    // Refresh view
                    RefreshView();
                }
                else
                {
                    // ._._._.
                    int cur = asciiTextBox.SelectionStart;
                    if ((cur % 1) == 1)
                    {
                        ++cur;
                    }
                    int pos = Math.Min((cur + 1) / 2, Count - 1);

                    if (Data == null)
                    {
                        Data = new byte[Count];
                        Modified = true;
                    }

                    Data[pos] = (byte)e.KeyChar;

                    cur += 2;

                    asciiTextBox.SelectionStart = Math.Min(cur, asciiTextBox.MaxLength - 1);
                    asciiTextBox.SelectionLength = 1;

                    // Refresh view
                    RefreshView();
                }
            }

            e.Handled = true;
        }

        #endregion

        #endregion
    }
}
